<?php
session_start();
require_once '../config/db.php';
require_once '../auth/auth.php';

// Check if user is logged in
if (!isset($_SESSION['user_id'])) {
    echo json_encode(['success' => false, 'message' => 'User not logged in']);
    exit();
}

// Get refund ID from request
$refund_id = $_GET['refund_id'] ?? null;
if (!$refund_id) {
    echo json_encode(['success' => false, 'message' => 'Refund ID is required']);
    exit();
}

try {
    // Get refund details
    $stmt = $pdo->prepare("
        SELECT 
            r.*,
            s.invoice_number as sale_id,
            DATE_FORMAT(r.refunded_at, '%Y-%m-%d %H:%i:%s') as refund_date,
            CONCAT('REF', LPAD(r.id, 6, '0')) as refund_id
        FROM refunds r
        JOIN sales s ON r.sale_id = s.id
        WHERE r.id = ?
    ");
    $stmt->execute([$refund_id]);
    $refund = $stmt->fetch(PDO::FETCH_ASSOC);

    if (!$refund) {
        echo json_encode(['success' => false, 'message' => 'Refund not found']);
        exit();
    }

    // Get refund items with product details
    $stmt = $pdo->prepare("
        SELECT 
            ri.*,
            p.name as item_name,
            CASE 
                WHEN ri.condition IN ('Damaged', 'Opened', 'Other') THEN ri.manual_amount
                ELSE (ri.unit_price * ri.quantity) - COALESCE(ri.discount, 0)
            END as total
        FROM refund_items ri
        JOIN products p ON ri.product_id = p.id
        WHERE ri.refund_id = ?
    ");
    $stmt->execute([$refund_id]);
    $items = $stmt->fetchAll(PDO::FETCH_ASSOC);

    // Process serial numbers for each item
    foreach ($items as &$item) {
        if (!empty($item['serial_numbers'])) {
            $item['serial_numbers_array'] = json_decode($item['serial_numbers'], true);
            $item['serial_numbers_display'] = implode(', ', $item['serial_numbers_array']);
        } else {
            $item['serial_numbers_array'] = [];
            $item['serial_numbers_display'] = 'No serial numbers';
        }
    }

    echo json_encode([
        'success' => true,
        'refund' => $refund,
        'items' => $items
    ]);
} catch (PDOException $e) {
    echo json_encode([
        'success' => false,
        'message' => 'Error fetching refund details: ' . $e->getMessage()
    ]);
} 